/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.juneau.server.config.repository;

import java.io.*;
import java.nio.file.*;
import java.util.*;

import org.apache.juneau.config.*;

@SuppressWarnings("javadoc")
public class GetConfiguration implements Command, GetValue<Map<String,ConfigItem>> {

	private static final String APPLICATION = "APPLICATION";
	private static final String PROJECT = "PROJECT";
	private static final String EXT = ".cfg";
	private static final String BAR = "/";

	private Map<String,ConfigItem> configs = new HashMap<>();

	private String project;
	private String branch;

	public GetConfiguration(String project, String branch) {
		this.branch = branch;
		this.project = project;
	}

	@Override
	public void execute() throws Exception {

		var config = Config.create().name("juneau-server-config.cfg").build();

		var pathStr = config.get("GitServer/pathLocal").orElse(null);

		var git = config.get("GitServer/gitRemote").orElse(null);

		var gitControl = new GitControl(pathStr, git);

		var path = new File(pathStr);

		if (path.isDirectory()) {
			gitControl.pullFromRepo();
		} else {
			gitControl.cloneRepo();
		}

		gitControl.branch(branch);
		gitControl.pullFromRepo();

		var fileDefaultStr = APPLICATION.toLowerCase().concat(EXT);
		var fileProjectStr = this.project.concat(EXT);

		var fileDefault = new File(pathStr.concat(BAR).concat(fileDefaultStr));
		if (fileDefault.exists()) {
			var lines = new String(Files.readAllBytes(fileDefault.toPath()));
			configs.put(APPLICATION, new ConfigItem(lines));
		}

		var fileProject = new File(pathStr.concat(BAR).concat(fileProjectStr));
		if (fileProject.exists()) {
			var linesProject = new String(Files.readAllBytes(fileProject.toPath()));
			configs.put(PROJECT, new ConfigItem(linesProject));
		}
	}

	@Override
	public Map<String,ConfigItem> get() {
		return configs;
	}
}